const AutoReplyModel = require('../models/AutoReplyModel');
const { encryptData } = require('../middleware/EncryptDecryptData');

exports.CreateAutoReplyForBrand = async (req, res) => {
  try {
    const { brand_id, auto_reply1, is_active1, auto_reply2, is_active2, ignore_users_type } = req.body;

    if (!brand_id) {
      return res.status(400).json({ message: 'Brand ID is required' });
    }

    await AutoReplyModel.create({ brand_id, auto_reply1, is_active1, auto_reply2, is_active2 });

    // Handle ignored users
    if (typeof ignore_users_type === 'object' && ignore_users_type !== null) {
      await Promise.all(Object.entries(ignore_users_type).map(([user_type, status]) => 
        AutoReplyModel.addOrUpdateIgnoredUser({ brand_id, user_type, status })
      ));
    }

    res.status(201).json(encryptData({ message: 'Auto-replies created successfully' }));
  } catch (error) {
    console.error('Error creating auto-replies:', error);
    res.status(500).json({ message: 'Internal server error' });
  }
};

exports.getAutoReplyForSpecificBrand = async (req, res) => {
  try {
    const { brand_id } = req.params;

    if (!brand_id) {
      return res.status(400).json({ message: 'Brand ID is required' });
    }

    const autoReply = await AutoReplyModel.findByBrandId(brand_id);
    if (!autoReply) {
      return res.status(404).json({ message: 'No auto-replies found for this brand' });
    }

    // Fetch ignored user types
    const ignore_users = await AutoReplyModel.getIgnoredUsers(brand_id);
    autoReply.ignore_users_type = Object.fromEntries(ignore_users.map(({ user_type, status }) => [user_type, status]));

    res.status(200).json(encryptData(autoReply));
  } catch (error) {
    console.error('Error fetching auto-replies:', error);
    res.status(500).json({ message: 'Internal server error' });
  }
};

exports.updateAutoReplyForSpecificBrand = async (req, res) => {
  try {
    const { brand_id, auto_reply1, is_active1, auto_reply2, is_active2, ignore_users_type } = req.body;

    if (!brand_id) {
      return res.status(400).json({ message: 'Brand ID is required' });
    }

    await AutoReplyModel.update({ brand_id, auto_reply1, is_active1, auto_reply2, is_active2 });

    if (typeof ignore_users_type === 'object' && ignore_users_type !== null) {
      await Promise.all(Object.entries(ignore_users_type).map(([user_type, status]) => 
        AutoReplyModel.addOrUpdateIgnoredUser({ brand_id, user_type, status })
      ));
    }

    res.status(200).json(encryptData({ message: 'Auto-replies updated successfully' }));
  } catch (error) {
    console.error('Error updating auto-replies:', error);
    res.status(500).json({ message: 'Internal server error' });
  }
};
