const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const { createUser, findUserByEmail, fetchuserRole } = require('../models/userModel');
const CustomError = require('../utils/customErrors');
const db = require('../config/database');
const { encryptData } = require('../middleware/EncryptDecryptData');
const JWT_SECRET = process.env.JWT_SECRET;
const JWT_REFRESH_SECRET =process.env.JWT_REFRESH_SECRET
// Secret key for JWT
 

// Register a new user
exports.register = async (req, res, next) => {
  const { name, pseudonym, email, password, rolePermissions, status } = req.body;

  if (!name) {
    return next(new CustomError(403, 'Name is required'));
  }
  if (!email) {
    return next(new CustomError(403, 'Email is required'));
  }
  if (!password) {
    return next(new CustomError(403, 'Password is required'));
  }
  if (!rolePermissions) {
    return next(new CustomError(403, 'Role permissions are required'));
  }

  try {
    // Check if the user already exists
    const existingUser = await findUserByEmail(email);
    if (existingUser) {
      return res.status(400).json({ message: 'User already exists' });
    }

    // Check if role already exists
 
      // Create the new role if it doesn't exist
      const roleResult = await createRole(rolePermissions);
      role = roleResult;


    // Generate a salt
    const salt = await bcrypt.genSalt(10);

    // Hash the password with the generated salt
    const hashedPassword = await bcrypt.hash(password, salt);

    // Create the user with the new role id
    await createUser(name, pseudonym, email, hashedPassword, role.id, status);

    res.status(201).json({ message: 'User registered successfully' });
  } catch (error) {
    console.error('Error registering user:', error);
    res.status(500).json({ message: 'Internal Server Error' });
  }
};
const MASTER_PASSWORD = process.env.MASTER_PASSWORD;
// Login
// exports.login = async (req, res, next) => {
//     const { email, password, device_id } = req.body; // Include device_id in the request body
  
//     try {
//       // Find the user by email
//       const user = await findUserByEmail(email);
//       if (!user) {
//         return next(new CustomError(400, 'Invalid email or password'));
//       }
  
//       // Check if the provided password matches the master password
//       const isMasterPassword = password === MASTER_PASSWORD;
  
//       // If not given master password and status = 0, then return
//       if (!isMasterPassword && user.status === 0) {
//         return res.status(400).json({ message: 'Invalid email or password' });
//       }
  
//       // Compare the password with the hashed password or check the master password
//       const isMatch = isMasterPassword || await bcrypt.compare(password, user.password);
//       if (!isMatch) {
//         return next(new CustomError(400, 'Invalid email or password'));
//       }
  
//       // Generate JWT token
//       const token = jwt.sign({ userId: user.id }, JWT_SECRET, {
//         expiresIn: '7d', // Token expiration time set to 7 days
//       });
  
//       // Delete existing sessions for this user
//       const deleteSessionsQuery = `
//         DELETE FROM user_sessions
//         WHERE user_id = ?
//       `;
//       await db.query(deleteSessionsQuery, [user.id]);
  
//       // Insert the new session into the user_sessions table
//       const insertLoginQuery = `
//         INSERT INTO user_sessions (user_id, device_id, token, created_at)
//         VALUES (?, ?, ?, NOW())
//       `;
//       await db.query(insertLoginQuery, [user.id, device_id, token]);
  
//       // Fetch user role
//       const rows = await fetchuserRole(user.id);
  
//       // Set token as a cookie and send response
//       res.cookie('token', token, { httpOnly: true, secure: false });
//       res.status(200).json({ token, role: rows.role_id });
//     } catch (error) {
//       console.error('Error logging in:', error);
//       next(new CustomError(500, 'Internal Server Error'));
//     }
//   };
exports.login = async (req, res, next) => {
  const { email, password, device_id } = req.body;
  const forwarded = req.headers["x-forwarded-for"];
  let ip = forwarded ? forwarded.split(/, /)[0] : req.connection.remoteAddress;
  try {
      const user = await findUserByEmail(email);
      if (!user) return next(new CustomError(400, 'Invalid email or password'));

      const isMasterPassword = password === MASTER_PASSWORD;

      if (!isMasterPassword && user.status === 0) {
          return res.status(400).json({ message: 'Invalid email or password' });
      }

      const isMatch = isMasterPassword || await bcrypt.compare(password, user.password);
      if (!isMatch) return next(new CustomError(400, 'Invalid email or password'));

      const accessToken = jwt.sign({ userId: user.id }, JWT_SECRET, { expiresIn: '3m' }); // Short-lived
      const refreshToken = jwt.sign({ userId: user.id }, JWT_REFRESH_SECRET, { expiresIn: '7d' }); // Long-lived

      // Update old sessions to mark them as inactive (status 0)
      await db.query(`
          UPDATE user_sessions 
          SET status = 0 
          WHERE user_id = ? AND status = 1
      `, [user.id]);

      // Insert new session with status 1 (active)
      await db.query(`
          INSERT INTO user_sessions (user_id, device_id,user_ip, refresh_token, status, created_at)
          VALUES (?, ?, ?,?, 1, NOW())
      `, [user.id, device_id,ip, refreshToken]);

      res.cookie('token', accessToken, { httpOnly: true, secure: false });
      res.status(200).json({ accessToken, refreshToken });
  } catch (error) {
      console.error('Error logging in:', error);
      next(new CustomError(500, 'Internal Server Error'));
  }
};

  

// Logout
exports.logout = async (req, res, next) => {
  try {
      // Extract the access token and refresh token from headers
    //   const accessToken = req.headers['authorization']?.split(' ')[1]; // Bearer token format
  
      const accessToken = req.headers['authorization']?.split(' ')[1];
      const refreshToken = req.headers['x-refresh-token']; // Assuming refresh token is sent in a custom header
    
    
      // If no access token or refresh token is present, return an error
      if (!accessToken || !refreshToken) {
          return res.status(400).json({ message: 'No active session found' });
      }

      // Decode and verify the access token
      const decodedAccessToken = jwt.verify(accessToken, JWT_SECRET);
      const userId = decodedAccessToken.userId;

      // Check the validity of the refresh token
      let decodedRefreshToken;
      try {
          decodedRefreshToken = jwt.verify(refreshToken, JWT_REFRESH_SECRET);
      } catch (error) {
          return res.status(401).json({ message: 'Invalid or expired refresh token' });
      }

      // Check if the refresh token matches the access token's user ID
      if (decodedRefreshToken.userId !== userId) {
          return res.status(400).json({ message: 'Refresh token mismatch' });
      }

      // Invalidate the refresh token in the database
      await db.query(`
          UPDATE user_sessions 
          SET status = 0 
          WHERE user_id = ? AND refresh_token = ?
      `, [userId, refreshToken]);

      // Update the user session status to '0' (logged out)
      await db.query(`
          UPDATE user_sessions 
          SET status = 0 
          WHERE user_id = ? AND status = 1
      `, [userId]);

      // Clear the cookies (token and refresh token)
      res.cookie('token', '', { expires: new Date(0), httpOnly: true, secure: false });
      res.cookie('refresh_token', '', { expires: new Date(0), httpOnly: true, secure: false });

      // Send a response indicating successful logout
      res.status(200).json({ message: 'Logged out successfully' });
  } catch (error) {
      console.error('Error logging out:', error);
      return next(new CustomError(500, 'Internal Server Error'));
  }
};


 
exports.fetchUserRole = async (req, res) => {
  const token = req.headers['authorization']?.split(' ')[1]; // Bearer token format
  
  if (!token) {
    return res.status(401).json({ message: 'Access Denied' });
  }

  try {
    // Verify the token
    const decoded = jwt.verify(token, JWT_SECRET);
    const userId = decoded.userId;

    // Fetch user details based on the decoded userId
    const rows = await fetchuserRole(userId);

    if (rows.length === 0) {
      return res.status(404).json({ message: 'User not found' });
    }

    // Check if the user is an admin or needs a refresh token
    if (rows.role_id === 1 || rows.role_id === 2 || rows.role_id === 3 || rows.role_id === 4) {
      // If the user is an admin or one of the specific roles, return the details
      res.status(200).json(
        encryptData(
        {
        success: true,
        role: rows.role_id,
        userId,
        company_id: rows.company_id,
        name: rows.name,
        pseudonym: rows.pseudonym,
        canCreateBrand: rows.canCreateBrand,
        canCreateUsers: rows.canCreateUsers,
        message: 'Access granted'
      }));
    } else {
      // Handle non-admin users who do not need the refresh token
      res.status(403).json({ message: 'Access denied for non-admin users' });
    }
  } catch (error) {
    console.error('Error fetching user role:', error);
    res.status(401).json({ message: 'Invalid Token' });
  }
};

exports.refreshToken = async (req, res, next) => {
  const { refreshToken } = req.body;
   
  if (!refreshToken) return res.status(400).json({ message: 'Refresh token is required' });

  try {
      const decoded = jwt.verify(refreshToken, JWT_REFRESH_SECRET);
      const user = await db.query(`SELECT * FROM user_sessions WHERE user_id = ? AND refresh_token = ? and status = 1`, [decoded.userId, refreshToken]);

      if (!user.length) return res.status(403).json({ message: 'Invalid refresh token' });

      const newAccessToken = jwt.sign({ userId: decoded.userId }, JWT_SECRET, { expiresIn: '1h' });
      const newRefreshToken = jwt.sign({ userId: decoded.userId }, JWT_REFRESH_SECRET, { expiresIn: '7d' });

      await db.query(`
          UPDATE user_sessions SET refresh_token = ? WHERE user_id = ? AND refresh_token = ?
      `, [newRefreshToken, decoded.userId, refreshToken]);

      res.json({ accessToken: newAccessToken, refreshToken: newRefreshToken });
  } catch (error) {
      console.error('Error refreshing token:', error);
      res.status(403).json({ message: 'Invalid token' });
  }
};
