const activeChatModel = require("../models/activeChat");
const chatRateModel = require("../models/chatRate");
const { fetchChatNotifications } = require("../models/chatModel");
// controllers/visitorController.js
const multer = require("multer");
const fs = require("fs");
const path = require("path");
const { v4: uuidv4 } = require("uuid");
const { encryptData } = require("../middleware/EncryptDecryptData");

// Create storage engine
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const publicKey = req.body.public_key; // Assume public_key is being passed in the request

    // Construct the directory path based on the public_key
    const dir = path.join(__dirname, `../public/uploads/chatData`);

    // Create the directory if it doesn't exist
    fs.promises.mkdir(dir, { recursive: true });

    // Set the directory for the file to be saved
    cb(null, dir);
  },
  filename: function (req, file, cb) {
    // Use a unique identifier (uuid) for the file name to avoid conflicts
    const uniqueName = `${uuidv4()}-${file.originalname}`;
    cb(null, uniqueName);
  },
});

const upload = multer({ storage: storage });

// Handle the file upload in the controller
exports.uploadFile = async (req, res) => {
  upload.single("file")(req, res, async function (err) {
    if (err instanceof multer.MulterError) {
      return res.status(500).json({ error: err.message });
    } else if (err) {
      return res.status(500).json({ error: "File upload failed." });
    }
    console.log("public key:", req.body.public_key);
    // Send back the file path or URL after the upload is successful
    const filePath = `/uploads/chatData/${req.file.filename}`;
    await activeChatModel.saveFileDetails(
      req.body.public_key,
      req.body.client_id,
      filePath,
      req.file.originalname
    );
    res.status(200).json({ filePath });
  });
};

exports.fetchClientMetadata = async (req, res) => {
  const { client_id } = req.params;

  try {
    const metadata = await activeChatModel.getVisitorMetadata(client_id);

    res.json(
      encryptData({
        client_id,
        join_time: metadata.created_at, // or use `join_time` if available
        os: metadata.os,
        ip: metadata.ip,
        country: metadata.country,
        browser: metadata.browser,
        city: metadata.city,
      })
    );
  } catch (error) {
    console.error("Error fetching client metadata:", error);
    res.status(500).json({ error: error.message });
  }
};

exports.startChat = async (req, res) => {
  const { public_key, client_id, user_id } = req.body;

  try {
    await activeChatModel.createActiveChat(public_key, client_id, user_id);
    res.status(200).json({ message: "Chat started successfully" });
  } catch (error) {
    res
      .status(500)
      .json({ message: "Error starting chat", error: error.message });
  }
};
exports.chatsNoti = async (req, res) => {
  const brand_id = req.params.brand_id;

  try {
    console.log("brand id.. :", brand_id);

    // Call the model function to get chat notifications
    const chatSessions = await fetchChatNotifications(brand_id);

    if (chatSessions.length === 0) {
      return res.status(404).json({ message: "No chat sessions found" });
    }

    res
      .status(200)
      .json(
        encryptData({
          message: "Chat sessions retrieved successfully",
          chatSessions,
        })
      );
  } catch (error) {
    console.error("Error fetching chat notifications:", error);
    res
      .status(500)
      .json({
        message: "Error fetching chat notifications",
        error: error.message,
      });
  }
};

exports.getActiveChats = async (req, res) => {
  try {
    const chats = await activeChatModel.getActiveChats();
    res.status(200).json(encryptData(chats));
  } catch (error) {
    res
      .status(500)
      .json({ message: "Error fetching active chats", error: error.message });
  }
};

// Fetch all messages by chatKey
exports.getMessagesByChatKey = async (req, res) => {
  try {
    const { chatKey, lastMessageId = null, isOldChat = false } = req.body;
    console.log("last message id", lastMessageId);
    console.log("is old chat", isOldChat);

    // Step 1: If fetching old chats, get old sessions first
    if (isOldChat) {
      const oldSessions = await activeChatModel.getOldSessionsByChatKey(
        chatKey
      );

      if (!oldSessions || oldSessions.length === 0) {
        return res
          .status(404)
          .json({ message: "No previous chat sessions found", oldChat: false });
      }

      // Fetch messages for all old sessions
      const oldMessages = await activeChatModel.getMessagesBySessionIds(
        oldSessions.map((s) => s.session_id),
        lastMessageId
      );
      const participants = await activeChatModel.getParticipantsForOldSessions(
        oldSessions.map((s) => s.session_id)
      );

      const formattedOldMessages = oldMessages.map((message) => {
        const sender = participants.find(
          (p) => p.user_id === message.sender_id
        );
        let senderType = "Unknown";
        if (sender) {
          senderType = sender.role === "admin" ? "admin" : "client";
        }
        console.log("all chats fetched.");

        return {
          from: sender ? sender.pseudonym : "Unknown",
          msg: message.message,
          type: senderType,
          msgtype: message.type,
          room: chatKey,
          sent_at: message.sent_at,
          messageId: message.message_id,
        };
      });

      return res
        .status(200)
        .json(encryptData({ messages: formattedOldMessages, oldChat: true }));
    } else {
      let activeSession;

      // Step 2: Fetch active session messages
      activeSession = await activeChatModel.getActiveSessionByChatKey(chatKey);

      if (!activeSession) {
        activeSession = await activeChatModel.getLastSessionByChatKey(chatKey);
      }

      if (!activeSession) {
        // Handle the case when no active or last session is found
        throw new Error(
          "No active or last session found for the given chat key."
        );
      }

      const { session_id } = activeSession; // Safe to destructure after the null check
      const participants = await activeChatModel.getParticipantsWithNames(
        session_id
      );
      const messages = await activeChatModel.getMessagesBySessionId(
        session_id,
        lastMessageId
      );

      const formattedMessages = messages.map((message) => {
        const sender = participants.find(
          (p) => p.user_id === message.sender_id
        );
        let senderType = "Unknown";
        if (sender) {
          senderType = sender.role === "admin" ? "admin" : "client";
        }

        return {
          from: sender ? sender.pseudonym : "Unknown",
          msg: message.message,
          type: senderType,
          msgtype: message.type,
          room: chatKey,
          sent_at: message.sent_at,
          messageId: message.message_id,
          is_seen: message.is_seen_msg,
        };
      });

      return res.status(200).json(
        encryptData({
          session: {
            chatKey,
            sessionId: session_id,
            participants,
          },
          messages: formattedMessages,
        })
      );
    }
  } catch (err) {
    console.error("Error fetching messages:", err);
    return res.status(500).json({ message: "Internal server error" });
  }
};
exports.SearchVisitorHistory = async (req, res) => {
  try {
    const { user_id, searchTerm } = req.body;
    const result = await activeChatModel.SearchVisitorHistory(
      user_id,
      searchTerm
    );
  } catch (error) {}
};
exports.getMessagesByChatSessionId = async (req, res) => {
  try {
    const { session_id } = req.body;

    // Check if session_id is provided
    if (!session_id) {
      return res
        .status(404)
        .json({ message: "session_id is required", status: 404 });
    }

    // Fetch participants for the session
    const participants = await activeChatModel.getParticipantsWithNames(
      session_id
    );

    if (!participants || participants.length === 0) {
      return res
        .status(404)
        .json({
          message: "No participants found for this session",
          status: 404,
        });
    }

    // Fetch messages for the session
    const messages = await activeChatModel.getMessagesBySessionId(session_id);

    // Format messages with sender information
    const formattedMessages = messages.map((message) => {
      const sender = participants.find((p) => p.user_id === message.sender_id);
      let senderType = "Unknown";

      if (sender) {
        senderType = sender.role === "admin" ? "admin" : "client";
      }

      return {
        from: sender ? sender.pseudonym : "Unknown",
        msg: message.message,
        type: senderType,
        msgtype: message.type,
        room: session_id, // Use session_id as room key here
        sent_at: message.sent_at,
        messageId: message.message_id,
        is_seen: message.is_seen,
      };
    });
    // Fetch rating based on session_id
    const rating = await chatRateModel.getRatingBySessionId(session_id);
    return res.status(200).json(
      encryptData({
        session: {
          sessionId: session_id,
          participants,
        },
        rating: rating ? rating.rating : 2,

        messages: formattedMessages,
        status: 200,
      })
    );
  } catch (err) {
    console.error("Error fetching messages:", err);
    return res.status(500).json({ message: "Internal server error" });
  }
};
