const { encryptData } = require('../middleware/EncryptDecryptData');
const ClientModel = require('../models/ClientModel');

const ClientController = {
    async VisitorHistory(req, res) {
        const { user_id } = req.body;
        const { page = 1, limit = 50 } = req.query; // Default page 1, limit 50
        
        try {
          const { rows: data, totalRecords } = await ClientModel.getClientData(user_id, page, limit);
          
          // Calculate total pages
          const totalPages = Math.ceil(totalRecords / limit);
       
          const responseData ={
            success: true,
            page: parseInt(page, 10),
            limit: parseInt(limit, 10),
            totalPages, // Return total pages
            totalRecords, // Total records for reference
            data // Paginated data
          }
          // Send response
          res.status(200).json(encryptData(responseData));
        } catch (err) {
          res.status(500).json({
            success: false,
            message: "Failed to fetch visitor history",
            error: err.message,
          });
        }
      },

      async updateSessionNotesAndTags(req, res) {
        const { session_id, notes, tags } = req.body;
    
        if (!session_id || notes === undefined || tags === undefined) {
          return res.status(400).json({
            success: false,
            message: "session_id, notes, and tags are required",
          });
        }
    
        try {
          const result = await ClientModel.updateSessionNotesAndTags(session_id, notes, tags);
          
          if (result.affectedRows === 0) {
            return res.status(404).json({
              success: false,
              message: "Session not found",
            });
          }
    
          res.status(200).json({
            success: true,
            message: "Notes and tags updated successfully",
          });
        } catch (err) {
          res.status(500).json({
            success: false,
            message: "Failed to update notes and tags",
            error: err.message,
          });
        }
      },
      
 // Controller method to change client status (block/unblock) and log the action
 async changeClientStatus(req, res) {
    const { blockValue, pseudonym, type } = req.body; // Expect blockValue (clientId or ip), pseudonym, and type in the request

    // Validate the incoming data
    if (!blockValue || !pseudonym || !type) {
        return res.status(400).json({ error: 'All fields (blockValue, pseudonym, type) are required.' });
    }

    try {
        // Check if the value is already blacklisted
        const isBlocked = await ClientModel.checkBlacklistStatus(type, blockValue);

        let message;
        if (isBlocked) {
            // If it's blacklisted (blocked), unblock it
            message = `${type === 'client' ? `Client ${blockValue}` : `IP ${blockValue}`} was unblocked by ${pseudonym}.`;
            // Unblock the client/IP
            await ClientModel.removeFromBlacklist(type, blockValue);
        } else {
            // If it's not blacklisted, block it
            message = `${type === 'client' ? `Client ${blockValue}` : `IP ${blockValue}`} was blocked by ${pseudonym}.`;
            // Block the client/IP
            await ClientModel.addToBlacklist(type, blockValue);
        }
        let value = type == 'client'?'client':'ip'
// Log the action (who blocked/unblocked the client with a generated message)
await ClientModel.createLog(blockValue, pseudonym, type, message);

        // Return success response
        res.json({ success: `${type === 'client' ? 'Client' : 'IP'} status successfully updated: ${isBlocked ? 'unblocked' : 'blocked'}.` ,type:type });
    } catch (error) {
        console.error('Error changing client status:', error);
        res.status(500).json({ error: 'An error occurred while changing client status.' });
    }
},




  // Controller method to get client data by chatKey
  async fetchClientData(req, res) {
    const { chatKey } = req.body;
    console.log("Here is chat key:", chatKey);
    
    if (!chatKey) {
        return res.status(400).json({ error: "ChatKey is required" });
    }

    try {
        // Fetch client data from the model using chatKey
        const clientData = await ClientModel.getClientDataByChatKey(chatKey);
        console.log("client data", clientData);
        
        if (!clientData) {
            return res.status(404).json({ error: "Client not found" });
        }

        // Parse tags if they are stored in JSON format
        if (clientData.tags) {
            clientData.tags = JSON.parse(clientData.tags);
        }

        // Check if client is blocked by clientId
        const isClientBlocked = await ClientModel.checkBlacklistStatus('client', clientData.client_id);

        // Fetch the IP associated with the client from the visitors table
        const visitorData = await ClientModel.getVisitorByClientId(clientData.client_id);
        console.log("visitor data",visitorData);
        
        let isIpBlocked = false;
        console.log("visitor ip", visitorData?.ip);
        
        if (visitorData && visitorData.ip) {
            // Check if the IP is blacklisted
            isIpBlocked = await ClientModel.checkBlacklistStatus('ip', visitorData.ip);
        }

        // Return the client data along with the block status
        res.json(encryptData({
            client: clientData,
            isClientBlocked,
            isIpBlocked
        }));
    } catch (error) {
        console.error('Error fetching client data:', error);
        res.status(500).json({ error: "An error occurred while fetching client data" });
    }
},


  // Fetch URLs for a visitor based on client_id
  async getVisitorUrls(req, res) {
    // const { client_id } = req.params;
    const {visitor_id,client_id } = req.body;
    console.log(visitor_id);
    console.log("visitor_id");
    try {
        if(visitor_id){
            console.log(visitor_id);
            console.log("visitor_id");
            
        
      // Fetch the join_time from visitor_session where status != 4
      const sessionData = await ClientModel.getVisitorSession(client_id,visitor_id);
      
      if (!sessionData) {
        return res.status(404).json({ message: "No active session found for the visitor" });
      }

      const { join_time } = sessionData;

      // Fetch all URLs from visitor table after join_time
      const urls = await ClientModel.getVisitorUrlsAfterJoinTime(client_id,visitor_id);

      res.status(200).json({ urls });
    }
    else{
        return res.status(400).json({ message: "Visitor ID is required" });
    }
    } catch (error) {
      console.error("Error fetching visitor URLs:", error);
      res.status(500).json({ message: "Server error" });
    }
  }, 
   async getUrlsByVisitorSession(req, res) {
    const { client_id, visitor_session_id } = req.body;
 console.log("fetching urls for history...");
 console.log('client id', client_id);
 console.log('session id ', visitor_session_id);
 
 
    try {
      // Fetch all URLs from visitor table after join_time
      const urls = await ClientModel.getUrlsByVisitorSession(visitor_session_id, client_id);

      res.status(200).json({ urls });
    } catch (error) {
      console.error("Error fetching visitor URLs:", error);
      res.status(500).json({ message: "Server error" });
    }
  },


// Controller function to fetch client metadata and total session counts
async fetchClientMetadata(req, res) {
    // const { client_id } = req.params;
    console.log(req.body);
    
    const {visitor_id,client_id} = req.body;
    console.log("visitor_id 333");
    console.log(visitor_id);
    try {
      // Fetch visitor metadata (join_time, left_time, os, ip, etc.)
      const metadata = await ClientModel.getVisitorMetadata(client_id,visitor_id);
     
  
      // Fetch total visitor sessions for the client
      const totalVisitorSessions = await ClientModel.getTotalVisitorSessions(client_id);
     
  
      // Fetch total chat sessions for the client
      const totalChatSessions = await ClientModel.getTotalChatSessions(client_id);
     
      // Respond with all the metadata and session counts
      res.json(encryptData({
        client_id,
        join_time: metadata.join_time || metadata.created_at, // Prefer join_time, fallback to created_at if unavailable
        left_time: metadata.left_time || null, // Include left_time if available, else null
        ip_health: metadata.ip_health || null,  
        os: metadata.os,
        ip: metadata.ip,
        country: metadata.country,
        browser: metadata.browser,
        city: metadata.city,
        total_visitor_sessions: totalVisitorSessions,
        total_chat_sessions: totalChatSessions
      }));
    } catch (error) {
      console.error("Error fetching client metadata:", error);
      res.status(500).json({ error: error.message });
    }
  },
  async updateIPStatus(req, res) {
    const { visitor_session_id, ip_health } = req.body; // Extract visitor session ID and new IP status

    try {
      // Update the IP health status in the database
      await ClientModel.updateIPHealth(visitor_session_id, ip_health);

      // Return the updated status
      res.status(200).json({
        success: true,
        message: ip_health === 1 ? 'IP marked as good' : 'IP marked as bad',
      });
    } catch (error) {
      res.status(500).json({
        success: false,
        message: 'Failed to update IP status',
        error: error.message,
      });
    }
  },
 
  // Function to handle client status request
  async getClientStatus(req, res){
    try {
      const { client_ids } = req.body; // Get the list of client_ids from the request body
  
      // Fetch status for each client_id
      const statuses = await Promise.all(
        client_ids.map(async (client_id) => {
          const statusEntry = await ClientModel.getLastVisitorStatus(client_id);
          return {
            client_id,
            status: statusEntry ? statusEntry.status : 'No session found', // Handle case where no session exists
            join_time: statusEntry ? statusEntry.join_time : null,
            left_time: statusEntry ? statusEntry.left_time : null,
            landing_url: statusEntry ? statusEntry.landing_url : null,
          };
        })
      );
  
      res.status(200).json({
        success: true,
        statuses,
      });
    } catch (error) {
      console.error('Error fetching client status:', error);
      res.status(500).json({
        success: false,
        message: 'Internal Server Error',
      });
    }
  },
  
//   async SearchVisitorHistoryController (req, res)  {
//     try {
//         const { user_id,searchTerm } = req.body; 
//         const result =await ClientModel.SearchVisitorHistory(searchTerm,user_id)
//         res.status(200).json({
//             success: true,
//             SearchData:result,
            
//           });
//     } catch (err) {
//         console.error('Error fetching client history search results:', err);
//         throw new Error('Failed to fetch client data');
//       }
//       },
      
async SearchVisitorHistoryController(req, res) {
    try {
      const { user_id, searchTerm, includeName, includeEmail, includeClientId,includeMsg ,connectedStatus,servedStatus, brandIds,currentPage } = req.body;
  
      // Call the model method with additional parameters
      const result = await ClientModel.SearchVisitorHistory(searchTerm, user_id, includeName, includeEmail, includeClientId,brandIds,includeMsg,servedStatus,connectedStatus,currentPage);
  
      res.status(200).json(encryptData({
        success: true,
        SearchData: result,
      }));
    } catch (err) {
      console.error('Error fetching client history search results:', err);
      res.status(500).json({ message: 'Failed to fetch client data' });
    }
  },
  
      async FilterVisitorHistoryByDateController (req, res)  {
    try {
        const { user_id,start_date,end_date } = req.body; 
        const result =await ClientModel.FilterVisitorHistoryByDate(user_id, start_date, end_date)
        res.status(200).json(encryptData({
            success: true,
            SearchData:result,
          }));
    } catch (err) {
        console.error('Error fetching client history search results:', err);
        throw new Error('Failed to fetch client data');
      }
      },
      async filterByBrands(req, res) {
        try {
          const user_id = req.user.id; // Assuming user ID is available in request object
          const brandIds = req.body.brandIds; // Expecting brand IDs in request body
    
          if (!Array.isArray(brandIds) || brandIds.length === 0) {
            return res.status(400).json({ message: 'Brand IDs must be an array and cannot be empty' });
          }
    
          const sessionRows = await ClientModel.filterByBrands(user_id, brandIds);
          const totalRecords = sessionRows.length > 0 ? sessionRows[0].total_records : 0;
    
          res.status(200).json(encryptData({ sessionRows, totalRecords }));
        } catch (error) {
          console.error('Error fetching visitor history by brands:', error);
          res.status(500).json({ message: 'Failed to fetch visitor history by brands' });
        }
      }

};
module.exports = ClientController;
