const mysql = require('mysql2/promise');
const fs = require('fs');
const fastCsv = require('fast-csv');

async function migrateData() {
    const connection = await mysql.createConnection({
        host: 'your_host',
        user: 'your_user',
        password: 'your_password',
        database: 'your_database'
    });

    try {
        console.log("Starting migration...");

        // Step 1: Modify the table
        await connection.execute(`
            ALTER TABLE user_shortcuts
            MODIFY COLUMN company_id VARCHAR(255) NOT NULL COLLATE latin1_swedish_ci,
            MODIFY COLUMN shortcuts VARCHAR(255) NOT NULL COLLATE latin1_swedish_ci
        `);

        console.log("Table structure updated.");

        // Step 2: Fetch existing shortcuts and users' company_id
        const [shortcutsRows] = await connection.execute(
            'SELECT id, user_id, shortcuts, created_at FROM user_shortcuts'
        );

        const [users] = await connection.execute(
            'SELECT id, company_id FROM users'
        );

        // Map user_id to company_id for quick lookup
        const userCompanyMap = {};
        users.forEach(user => {
            userCompanyMap[user.id] = user.company_id;
        });

        let newData = [];

        for (let row of shortcutsRows) {
            let shortcutsArray = JSON.parse(row.shortcuts || '[]'); // Assuming JSON format

            for (let shortcut of shortcutsArray) {
                newData.push([
                    row.user_id,
                    userCompanyMap[row.user_id] || 'default_company_id', // Default if missing
                    0, // shortcut_type (default private)
                    null, // created_by
                    null, // updated_by
                    null, // deleted_by
                    shortcut, // shortcuts as string
                    1, // status (default active)
                    row.created_at,
                    null // updated_at
                ]);
            }
        }

        if (newData.length > 0) {
            console.log(`Saving ${newData.length} records to CSV...`);

            // Step 3: Write data to CSV file before inserting into DB
            const ws = fs.createWriteStream("shortcuts_backup.csv");
            fastCsv
                .write(newData, { headers: ["user_id", "company_id", "shortcut_type", "created_by", "updated_by", "deleted_by", "shortcuts", "status", "created_at", "updated_at"] })
                .pipe(ws)
                .on("finish", async () => {
                    console.log("Data saved to CSV successfully.");

                    console.log(`Inserting ${newData.length} new records...`);
                    
                    // Step 4: Insert updated records into the database
                    await connection.query(`
                        INSERT INTO user_shortcuts 
                        (user_id, company_id, shortcut_type, created_by, updated_by, deleted_by, shortcuts, status, created_at, updated_at) 
                        VALUES ?
                    `, [newData]);

                    console.log("Data migration completed successfully.");
                });

        } else {
            console.log("No data to insert.");
        }

    } catch (error) {
        console.error("Migration failed:", error);
    } finally {
        await connection.end();
    }
}

// Run migration
migrateData();
