const db = require('../config/database');
const { encryptData } = require('../middleware/EncryptDecryptData');

// Create a new alert message
exports.createAlert = async (req, res) => {
  try {
    const { msg, user_id, is_active,type, expired_at } = req.body;
    const { role, userId } = req.user;

    // Only admins or users creating alerts for themselves are allowed
    if (role !== 1 && user_id !== userId) {
      return res.status(403).json({ message: 'Unauthorized' });
    }

    const deactivateQuery = 'UPDATE alerts SET is_active = 0 WHERE type = ? ';
    await db.query(deactivateQuery, ['targeted']);
 
    const formattedExpiredAt = expired_at
      ? `${expired_at} 14:59:59` // Append 11:59 PM as the default time
      : null;

    const query = 'INSERT INTO alerts (msg, user_id, is_active,type, expired_at) VALUES (?, ?, ?,?, ?)';
    const [result] = await db.query(query, [msg, user_id, is_active || 0,type, formattedExpiredAt || null]);

    res.status(201).json({ message: 'Alert created successfully', alertId: result.insertId });
  } catch (error) {
    console.error('Error creating alert:', error);
    res.status(500).json({ message: 'Error creating alert', error });
  }
};

// Publish (activate) an alert and set expiration date
exports.publishAlert = async (req, res) => {
  try {
    const { id } = req.params;
    const { expired_at } = req.body;
    if (expired_at) {
      // First, deactivate all other alerts with the same 'type'
      const deactivateQuery = 'UPDATE alerts SET is_active = 0 WHERE type = ? ';
      await db.query(deactivateQuery, ['targeted']);
    }
    const formattedExpiredAt = expired_at
    ? `${expired_at} 14:59:59` // Append 11:59 PM as the default time
    : null;

    const query = 'UPDATE alerts SET is_active = 1, published_at = CURRENT_TIMESTAMP, expired_at = ? WHERE id = ?';
    const [result] = await db.query(query, [formattedExpiredAt, id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Alert not found' });
    }

    res.status(200).json({ message: 'Alert published successfully' });
  } catch (error) {
    console.error('Error publishing alert:', error);
    res.status(500).json({ message: 'Error publishing alert', error });
  }
};

// Deactivate an alert
exports.deactivateAlert = async (req, res) => {
  try {
    const { id } = req.params;

    const query = 'UPDATE alerts SET is_active = 0 WHERE id = ?';
    const [result] = await db.query(query, [id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Alert not found' });
    }

    res.status(200).json({ message: 'Alert deactivated successfully' });
  } catch (error) {
    console.error('Error deactivating alert:', error);
    res.status(500).json({ message: 'Error deactivating alert', error });
  }
};
exports.deleteAlert = async (req, res) => {
  try {
    const { id } = req.params;

    // Corrected SQL query syntax
    const query = 'DELETE FROM alerts WHERE id = ?';
    const [result] = await db.query(query, [id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Alert not found' });
    }

    res.status(200).json({ message: 'Alert deleted successfully' });
  } catch (error) {
    console.error('Error deleting alert:', error);
    res.status(500).json({ message: 'Error deleting alert', error });
  }
};


// Get all active alerts
exports.getActiveAlerts = async (req, res) => {
  try {
    // Mark expired alerts as inactive
    const markExpiredQuery = `
      UPDATE alerts 
      SET is_active = 0 
      WHERE is_active = 1 AND expired_at IS NOT NULL AND expired_at <= CURRENT_TIMESTAMP
    `;
    await db.query(markExpiredQuery);

    // Fetch the last added active alert
    const fetchActiveQuery = `
      SELECT msg, type as alertType 
      FROM alerts 
      WHERE is_active = 1 AND ( expired_at > CURRENT_TIMESTAMP)
      ORDER BY created_at DESC 
      LIMIT 1
    `;
    const [alerts] = await db.query(fetchActiveQuery);

    res.status(200).json(encryptData({
      message: 'Active alert fetched successfully',
      alert: alerts.length ? alerts[0] : null,
    }));
  } catch (error) {
    console.error('Error fetching active alerts:', error);
    res.status(500).json({ message: 'Error fetching active alerts', error });
  }
};


// Get all alerts grouped by active and inactive
exports.getAlertsByCategory = async (req, res) => {
  try {
    const queryTargeted = 'SELECT * FROM alerts WHERE type = "targeted" ORDER BY created_at DESC';
    const queryOneTime = 'SELECT * FROM alerts WHERE type = "one-time" ORDER BY created_at DESC';

    const [targetedAlerts] = await db.query(queryTargeted);
    const [oneTimeAlerts] = await db.query(queryOneTime);

    res.status(200).json(
        encryptData(
        {
      message: 'Alerts fetched by category successfully',
      targetedAlerts,
      oneTimeAlerts,
    }));
  } catch (error) {
    console.error('Error fetching alerts by category:', error);
    res.status(500).json({ message: 'Error fetching alerts by category', error });
  }
};
