const db = require('../config/database');

const createActiveChat = async (public_key, client_id, user_id) => {
    try {
        const [result] = await db.query(
            'INSERT INTO active_chats (public_key, client_id, user_id) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE user_id = VALUES(user_id)',
            [public_key, client_id, user_id]
        );
        return result;
    } catch (error) {
        throw new Error('Error creating chat: ' + error.message);
    }
};

exports.fetchChatNotifications = async (brand_id) => {
    try {
      console.log("brand id.. :", brand_id);
  
      // Query to fetch the public_key from the brands table
      const [brands] = await db.query('SELECT public_key FROM brands WHERE id = ?', [brand_id]);
  
      if (!brands || brands.length === 0) {
        console.log("brand not found");
        throw new Error('Brand not found');
      }
  
      const public_key = brands[0].public_key; // Use brands[0] to access the first element
      console.log('public_key:', public_key);
  
      // Query to fetch chat sessions using the public_key
      const [chatSessions] = await db.query('SELECT * FROM chat_sessions WHERE public_key = ?', [public_key]);
  
      return chatSessions;
    } catch (error) {
      console.error('Error fetching chat notifications:', error);
      throw error; // Rethrow the error to be handled by the controller
    }
  };
const getActiveChats = async () => {
    try {
        const [rows] = await db.query('SELECT * FROM active_chats');
        return rows;
    } catch (error) {
        throw new Error('Error fetching chats: ' + error.message);
    }
};
// Fetch pseudonym from users table
const getUserNameById = async (userId) => {
    const query = `SELECT pseudonym FROM users WHERE id = ? LIMIT 1`;
  
    try {
      const [results] = await db.query(query, [userId]);
      return results.length ? results[0].pseudonym : null;
    } catch (err) {
      throw new Error(`Error fetching user pseudonym name: ${err.message}`);
    }
  };
  
  // Fetch client name from client table
  const getClientNameById = async (clientId) => {
    const query = `SELECT name FROM client WHERE client_id = ? LIMIT 1`;
  
    try {
      const [results] = await db.query(query, [clientId]);
      return results.length ? results[0].name : null;
    } catch (err) {
      throw new Error(`Error fetching client name: ${err.message}`);
    }
  };
  const getActiveSessionByChatKey = async (chatKey) => {
    const query = `SELECT * FROM chat_sessions WHERE chat_key = ? AND is_active = 1 LIMIT 1`;
  
    try {
      const [results] = await db.query(query, [chatKey]);
      return results[0]; // Expecting one session (or null)
    } catch (err) {
      throw new Error(`Error fetching active session: ${err.message}`);
    }
  };
  const getLastSessionByChatKey = async (chatKey) => {
    const query = `SELECT * FROM chat_sessions WHERE chat_key = ? ORDER BY chat_initiated_time DESC LIMIT 1`;
  
    try {
      const [results] = await db.query(query, [chatKey]);
      return results[0]; // Return the last session (or null if no session exists)
    } catch (err) {
      throw new Error(`Error fetching last session: ${err.message}`);
    }
  };
  
  
 // Fetch admin's pseudonym and client's name by session_id
const getParticipantsWithNames = async (sessionId) => {
    const participantsQuery = `SELECT * FROM chat_participants WHERE session_id = ?`;
  
    try {
      const [participants] = await db.query(participantsQuery, [sessionId]);
  
      // For each participant, fetch the name (admin -> users table, client -> client table)
      const participantsWithNames = await Promise.all(
        participants.map(async (participant) => {
          let pseudonym = 'Unknown';
  
          if (participant.role === 'admin') {
            // Fetch pseudonym from users table
            pseudonym = await getUserNameById(participant.user_id);
          } else if (participant.role === 'client') {
            // Fetch client name from client table using client_id
            pseudonym = await getClientNameById(participant.user_id);
          }
  
          return {
            ...participant,
            pseudonym: pseudonym || 'Unknown',  // Add pseudonym or 'Unknown' if not found
          };
        })
      );
  
      return participantsWithNames;
    } catch (err) {
      throw new Error(`Error fetching participants with names: ${err.message}`);
    }
  };
  const getMessagesBySessionId = async (sessionId) => {
    const query = `
      SELECT 
    cm.*, 
    CASE 
        WHEN cs.message_id IS NOT NULL THEN 1 
        ELSE 0 
    END AS is_seen_msg,
    cs.participant_type
FROM chat_messages cm
LEFT JOIN chat_seens cs ON cm.message_id = cs.message_id
WHERE cm.session_id = ?
ORDER BY cm.sent_at DESC;
    `;

    try {
        const [results] = await db.query(query, [sessionId]);
        return results;
    } catch (err) {
        throw new Error(`Error fetching messages: ${err.message}`);
    }
};

  const getOldSessionsByChatKey = async (chatKey) => {
    console.log("chat key");

    
    const query = `
      SELECT session_id 
      FROM chat_sessions 
      WHERE chat_key = ? AND is_active = 0
    `;
  
    try {
      const [oldSessions] = await db.query(query, [chatKey]);
      return oldSessions;
    } catch (err) {
      throw new Error(`Error fetching old sessions: ${err.message}`);
    }
  };
  const getMessagesBySessionIds = async (sessionIds, lastMessageId = null) => {
    let query = `SELECT * FROM chat_messages WHERE session_id IN (?)`;
const params = [sessionIds];

console.log("last message id", lastMessageId);

// Only add this condition if lastMessageId is valid
if (lastMessageId) {
  query += ` AND message_id < ?`;
  params.push(lastMessageId);
}

// Fetch the last 20 messages (older than lastMessageId)
query += ` ORDER BY sent_at DESC LIMIT 20`; // Limit to 20 messages

    try {
      const [messages] = await db.query(query, params);
      return messages;
    } catch (err) {
      throw new Error(`Error fetching old messages: ${err.message}`);
    }
  };
  const getParticipantsForOldSessions = async (sessionIds) => {
    const query = `SELECT * FROM chat_participants WHERE session_id IN (?)`;
  
    try {
      const [participants] = await db.query(query, [sessionIds]);
  
      // Fetch participant names (admin from users table, client from client table)
      const participantsWithNames = await Promise.all(
        participants.map(async (participant) => {
          let pseudonym = 'Unknown';
  
          if (participant.role === 'admin') {
            pseudonym = await getUserNameById(participant.user_id);
          } else if (participant.role === 'client') {
            pseudonym = await getClientNameById(participant.user_id);
          }
  
          return {
            ...participant,
            pseudonym: pseudonym || 'Unknown',
          };
        })
      );
  
      return participantsWithNames;
    } catch (err) {
      throw new Error(`Error fetching participants for old sessions: ${err.message}`);
    }
  };
      
 

  const saveFileDetails = async (publicKey,client_id, filePath, originalName) => {
    const query = `INSERT INTO uploaded_files (public_key,client_id, file_path, original_name) VALUES (?, ?, ?, ?)`;
    return db.query(query, [publicKey,client_id, filePath, originalName]);
  };
  
 
  
module.exports = {
    createActiveChat,
    getActiveChats,
    getMessagesBySessionId,
    saveFileDetails,
    getActiveSessionByChatKey,
    getParticipantsWithNames,
    getParticipantsForOldSessions,
    getOldSessionsByChatKey,
    getMessagesBySessionIds,
    getLastSessionByChatKey,
};
