// models/userModel.js
const db = require('../config/database');
const bcrypt = require('bcryptjs');
const crypto = require('crypto');
// Create a new user

const checkEmailExists = async (email) => {
    const [rows] = await db.query('SELECT 1 FROM users WHERE email = ?', [email]);
    return rows.length > 0;
  };
  
 
  const createUser = async (name, pseudonym, email, password, hrm_id, designation, role, company_id, brand_id, status, created_by, canCreateUsers, canCreateBrand) => {
    try {
      // Hash the password
      const hashedPassword = await bcrypt.hash(password, 10);
  
      // Create the chat_hash by hashing the hrm_id using MD5
      const chat_hash = crypto.createHash('md5').update(hrm_id).digest('hex');
  
      // Start a transaction
      await db.query('START TRANSACTION');
  
      // Insert into the users table
      const [result] = await db.query(
        'INSERT INTO users (name, pseudonym, email, password, hrm_id, chat_hash, role_id, designation, company_id, created_by, status, canCreateUsers, canCreateBrand) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
        [name, pseudonym, email, hashedPassword, hrm_id, chat_hash, role, designation, company_id, created_by, status, canCreateUsers, canCreateBrand]
      );
  
      const userId = result.insertId; // Get the inserted user ID
  
      // Insert into the brand_assignees table
      if (brand_id) {
        await db.query(
          'INSERT INTO brand_assignees (brand_id, user_id) VALUES (?, ?)',
          [brand_id, userId]
        );
      }
  
     // Insert the user ID into the user_tones table with default values for other columns
await db.query(
    'INSERT INTO user_tones (user_id) VALUES (?)',
    [userId]
  );
  
  
      // Commit the transaction
      await db.query('COMMIT');
  
      return { userId, success: true };
    } catch (error) {
      // Rollback the transaction in case of an error
      await db.query('ROLLBACK');
      console.error('Error creating user:', error);
      return { success: false, error: error.message };
    }
  };
  
 

// Find a user by email
const findUserByEmail = async (email) => {
  const [rows] = await db.query('SELECT * FROM users WHERE email = ?', [email]);
  return rows[0];
};

// Find a user by ID
const findUserById = async (id) => {
    const [rows] = await db.query('SELECT * FROM users WHERE id = ?', [id]);
    if (rows.length > 0) {
      const { password, ...userWithoutPassword } = rows[0]; // Exclude password
      return userWithoutPassword;
    }
    return null;
  };
  
const getProfile = async (id) => {
    const [rows] = await db.query('SELECT name,pseudonym,designation,email,profile FROM users WHERE id = ?', [id]);
    return rows[0];
  };
  const getUsersRelatedToCompanyModel = async (user_id) => {
    try {
        console.log(user_id);
        
        // Query to get the company_id for the given user_id
        const [companyResult] = await db.query(
            "SELECT company_id  FROM users WHERE id = ? LIMIT 1",
            [user_id]
        );

        if (companyResult.length === 0) {
            throw new Error("User not found or invalid user_id.");
        }
        console.log("company id");
        console.log(companyResult[0].company_id);
        
        const company_id = companyResult[0].company_id;

        // Query to fetch all users with the same company_id
        const [users] = await db.query(
            "SELECT id AS user_id, company_id, name FROM users WHERE company_id = ?",
            [company_id]
        );

        return users; // Returns an array of users in the same company
    } catch (error) {
        console.error("Error fetching users related to the company:", error.message);
        throw error;
    }
};
const findUserByCompId = async (id) => {
  const query = `
  SELECT 
    u.id, 
    u.pseudonym, 
    u.email, 
    u.role_id, 
    u.designation, 
    u.company_id, 
    u.created_at,
    u.status,
    u.name,
     
    creator.pseudonym AS created_by_pseudonym,
    c.name AS company_name
  FROM users u
  LEFT JOIN users creator ON u.created_by = creator.id
  LEFT JOIN companies c ON u.company_id = c.id
  WHERE u.company_id = ?
  order by c.id
`;
[users] = await db.query(query, [id]);
  return users;
};

// Find all users
const findAllUsers = async () => {
  const query = `
    SELECT 
      u.id, 
      u.pseudonym, 
      u.email, 
      u.role_id, 
      u.designation,
      u.company_id, 
      u.created_at,
      u.status,
      u.name,
    
      creator.pseudonym AS created_by_pseudonym,
      c.name AS company_name
    FROM users u
    LEFT JOIN users creator ON u.created_by = creator.id
    LEFT JOIN companies c ON u.company_id = c.id
    order by c.id

  `;

  const [rows] = await db.query(query);
  return rows;
};
 


// Update user information
const updateUser = async (id, name, pseudonym, email, password, role,designation, company_id, status) => {
  const hashedPassword = password ? await bcrypt.hash(password, 10) : undefined;
  const [result] = await db.query(
    'UPDATE users SET name = ?, pseudonym = ?, email = ?, password = COALESCE(?, password), role = ?, designation = ?, company_id = ?, status = ? WHERE id = ?',
    [name, pseudonym, email, hashedPassword, role,designation, company_id, status, id]
  );
  return result;
};

const updateProfileModel = async (name, pseudonym, email, designation, userId, profileImage) => {

    let query = 'UPDATE users SET name = ?, pseudonym = ?, email = ?, designation = ?';
    let values = [name, pseudonym, email, designation];
  
    if (profileImage) {
      query += ', profile = ?';
      let profilePath = 'uploads/user_Profiles/'+profileImage;
      values.push(profilePath);
    }
  
    query += ' WHERE id = ?';
    values.push(userId);
  
    const [result] = await db.query(query, values);
    return result;
  };
  
const updateProfilePassword = async (hashedPassword, userId) => {
    const [result] = await db.query(
      'UPDATE users SET password = ? WHERE id = ?',
      [hashedPassword, userId]
    );
    return result;
  };
  
  
  
// Change user status (e.g., enable/disable)
const updateStatus = async (id, status) => {
  const [result] = await db.query('UPDATE users SET status = ? WHERE id = ?', [status, id]);
  return result;
};
// Function to fetch users by brand ID
const findUsersByBrandId = async (brandId) => {
  const [userIds] = await db.query('SELECT user_id FROM brand_assignees WHERE brand_id = ? and status = ?', [brandId,1]);
  
  
  if (userIds.length === 0) {
    return [];
  }
  
  const ids = userIds.map(row => row.user_id);
  const [users] = await db.query('SELECT id, name, pseudonym,designation, email, role_id, created_at, status FROM users WHERE id IN (?)', [ids]);
  
  return users;
};
const fetchuserRole  = async(userId)=>{
    // Fetch user data from the database
    const [rows] = await db.query('SELECT role_id,name,pseudonym, company_id ,canCreateUsers,canCreateBrand, pseudonym FROM users WHERE id = ?', [userId]);
    return rows[0];
}
module.exports = {
  createUser,
  findUserByEmail,
  findUserById,
  findAllUsers,
  getProfile,
  updateUser,
  updateProfileModel,
  checkEmailExists,
  updateProfilePassword, // Export the new function
  updateStatus,findUsersByBrandId,fetchuserRole,findUserByCompId,getUsersRelatedToCompanyModel
};
