const connection = require('../config/database');
const geoip = require("geoip-lite"); // Import the geoip-lite library
const countries = require("countries-list").countries;

const handleDisconnectTimeout = (client_id, public_key, connection, io, adminSockets, activeChats, connectedClients, visitorState, notifyAssignedAdmins) => {
    console.log("Handling disconnection for:", client_id, public_key);
    const chatKey = `${public_key}_${client_id}`;
    const disconnectTime = new Date();

    const getSessionQuery = `
        SELECT session_id 
        FROM chat_sessions 
        WHERE client_id = ? AND is_active = 1 LIMIT 1
    `;

    connection.query(getSessionQuery, [client_id], (err, sessionResults) => {
        if (err) {
            console.error("Error fetching session:", err);
            return;
        }

        if (sessionResults.length > 0) {
            const sessionId = sessionResults[0].session_id;
            const insertMessageQuery = `
                INSERT INTO chat_messages (session_id, sender_id, message, type, sent_at)
                VALUES (?, ?, ?, ?, NOW())
            `;

            connection.query(insertMessageQuery, [sessionId, client_id, 'visitor left the chat', "comment"], (err, result) => {
                if (err) {
                    console.error("Error inserting message:", err);
                    return;
                }

                if (result?.insertId) {
                    io.to(chatKey).emit("chat message", {
                        from: client_id,
                        msg: 'visitor left the chat',
                        type: "client",
                        msgtype: "comment",
                        room: chatKey,
                        sent_at: new Date().toISOString(),
                        messageId: result.insertId,
                    });
                }
            });
        }
    });

    const updateStatusQuery = `
        UPDATE visitor_session 
        SET status = 4, left_time = ? 
        WHERE client_id = ? AND status != 4
    `;

    connection.query(updateStatusQuery, [disconnectTime, client_id], (err) => {
        if (err) {
            console.error("Error updating visitor session status:", err);
        } else {
            console.log(`Visitor session status set to 4 for client ${client_id}`);
            delete activeChats[chatKey];
            delete connectedClients[client_id];
            delete visitorState[client_id];
        }
    });

    const updateChatSessionQuery = `
        UPDATE chat_sessions 
        SET is_active = 0 
        WHERE client_id = ? AND is_active = 1
    `;

    connection.query(updateChatSessionQuery, [client_id], (err) => {
        if (err) {
            console.error("Error updating chat session status:", err);
        } else {
            console.log(`Chat session set to inactive for client ${client_id}`);
        }
    });
};

const handleStatusChange = (client_id, public_key, newStatus, connection, visitorState, notifyUser) => {
    if (visitorState[client_id]?.statusLock) return;
    visitorState[client_id].statusLock = true;

    const updateStatusQuery = `
        UPDATE visitor_session 
        SET status = ? 
        WHERE client_id = ? AND status != 4
    `;

    connection.query(updateStatusQuery, [newStatus, client_id], (err) => {
        visitorState[client_id].statusLock = false;
        if (err) {
            console.error("Error updating visitor session:", err);
        } else {
            console.log(`Visitor status updated to ${newStatus} for client ${client_id}`);
            notifyUser(public_key, newStatus, "Visitor status change");
        }
    });
};
const notifyAssignedAdmins = (
    public_key,
    connection,
    adminSockets,
    emitEvent,
    emitData
  ) => {
    const assignedAdminsQuery = `
    SELECT 
        user_id 
    FROM 
        brand_assignees 
    WHERE 
        brand_id = (SELECT id FROM brands WHERE public_key = ?)
        AND status = 1
  `;
  
  
    connection.query(assignedAdminsQuery, [public_key], (err, adminResults) => {
      if (err) {
        console.error("Error fetching assigned admins:", err);
        return;
      }
      console.log("Admins fetched");
  
      // Notify each assigned admin with the specific event and data
      adminResults.forEach((row) => {
        console.log(row);
        const adminSocket = adminSockets[row.user_id];
  
        if (adminSocket) {
          adminSocket.emit(emitEvent, emitData);
          console.log(
            `Admin ${row.user_id} notified with event '${emitEvent}' and data:`,
            emitData
          );
        }
      });
    });
  };
  //more 
  

const notifyChatParticipants = (
    public_key,
    connection,
    adminSockets,
    emitEvent,
    emitData
  ) => {
    const assignedAdminsQuery = `
      SELECT user_id 
      FROM brand_assignees 
      WHERE brand_id = (SELECT id FROM brands WHERE public_key = ?)
      AND status = 1
    `;
  
    connection.query(assignedAdminsQuery, [public_key], (err, adminResults) => {
      if (err) {
        console.error("Error fetching assigned admins:", err);
        return;
      }
  
      console.log("Admins fetched");
  
      // Check if validParticipants is empty or not
      const notifyAllAdmins = emitData.participants.length === 0;
      console.log("notifyAllAdmins:", notifyAllAdmins);
  
      adminResults.forEach((row) => {
        const adminId = Number(row.user_id); // Convert adminId to a number
        const adminSocket = adminSockets[adminId];
  
        // Ensure the admin has a valid socket connection before emitting
        if (!adminSocket) {
          console.log(`No socket connection found for admin ${adminId}`);
          return; // Skip this admin if no socket is found
        }
  
        // Determine whether to notify this admin
        const isParticipant = notifyAllAdmins
          ? true
          : emitData.participants.includes(adminId);
  
        // Notify all admins if validParticipants is empty, otherwise only notify participants
        adminSocket.emit(emitEvent, {
          ...emitData,
          isParticipant, // Send the participant flag to the frontend
        });
        console.log(
          `Admin ${adminId} notified with event '${emitEvent}', participant: ${isParticipant}`
        );
      });
    });
  };
  
  
  // Fetch user name by user ID using callback
  const getUserNameById = (userId, callback) => {
    try {
      console.log(`Fetching user name for userId: ${userId}`);
  
      const query = `SELECT real_name FROM users WHERE id = ? LIMIT 1`;
  
      connection.query(query, [userId], (err, results) => {
        if (err) {
          console.error(`Database error while fetching user real name: ${err.message}`);
          return callback(
            new Error(`Error fetching user real name: ${err.message}`),
            null
          );
        }
  
        const realName = results.length ? results[0].real_name : null;
  
        if (!realName) {
          console.warn(`No user found with userId: ${userId}`);
        }
  
        callback(null, realName);
      });
    } catch (error) {
      console.error(`Unexpected error in getUserNameById: ${error.message}`);
      callback(new Error(`Unexpected error: ${error.message}`), null);
    }
  };
  
  
  // Fetch client name by client ID using callback
  const getClientNameById = (clientId, callback) => {
    const query = `SELECT name FROM client WHERE client_id = ? LIMIT 1`;
  
    connection.query(query, [clientId], (err, results) => {
      if (err) {
        return callback(
          new Error(`Error fetching client name: ${err.message}`),
          null
        );
      }
      const clientName = results.length ? results[0].name : null;
      callback(null, clientName);
    });
  };
  
  // Fetch participants by session ID and their names using callbacks
  function getParticipantsWithNames(sessionId, callback) {
    const participantsQuery = `SELECT * FROM chat_participants WHERE session_id = ?`;
  
    connection.query(participantsQuery, [sessionId], (err, participants) => {
      if (err) {
        return callback(err, null);
      }
  
      const participantsWithNames = [];
      let count = 0;
  
      participants.forEach((participant) => {
        if (participant.role === "admin") {
          // Fetch admin name
          getUserNameById(participant.user_id, (nameErr, realName) => {
            participantsWithNames.push({
              ...participant,
              real_name: realName || "Unknown",
            });
            count++;
            if (count === participants.length) {
              callback(null, participantsWithNames);
            }
          });
        } else if (participant.role === "client") {
          // Fetch client name
          getClientNameById(participant.user_id, (nameErr, realName) => {
            participantsWithNames.push({
              ...participant,
              real_name: realName || "Unknown",
            });
            count++;
            if (count === participants.length) {
              callback(null, participantsWithNames);
            }
          });
        }
      });
    });
  }
  
  // Fetch messages by session ID
  function getMessagesBySessionId(sessionId, callback) {
    const query = `SELECT * FROM chat_messages WHERE session_id = ? ORDER BY sent_at ASC`;
  
    connection.query(query, [sessionId], (err, messages) => {
      if (err) {
        return callback(err, null);
      }
  
      callback(null, messages);
    });
  }
  
  
  
  const getGeolocationData = async (ipAddress) => {
    // Get geolocation data
  
    const geo = geoip.lookup(ipAddress);
  
    if (geo) {
      const { country, region, city } = geo;
  
      // Get full country name
      const fullCountryName = countries[country]?.name || country;
  
      return {
        countryName: fullCountryName,
        cityName: city || "Unknown",
      };
    } else {
      console.log("Geolocation data not found.");
      return {
        countryName: "Unknown",
        cityName: "Unknown",
      };
    }
  };
  // Function to check rating and emit the event if a rating exists
  function checkAndEmitRating(socket, sessionId,client_id) {
    const checkRatingQuery = `SELECT rating FROM chat_rate WHERE chat_id = ?`;
    connection.query(checkRatingQuery, [sessionId], (err, ratingResults) => {
      if (err) {
        console.error("Error checking rating:", err);
        return;
      }
  
      if (ratingResults.length > 0 && ratingResults[0].rating !== null) {
        // Emit socket event with the rating
        socket.emit("chat rating", {
          session_id: sessionId,
          rating: ratingResults[0].rating,
          client_id:client_id,
        });
        console.log(`Rating emitted for session ${sessionId}:`, ratingResults[0].rating);
      } else {
        console.log(`No rating found for session ${sessionId}`);
        socket.emit("chat rating", {
          session_id: sessionId,
          rating: 3,
          client_id:client_id,
        });
      }
    });
  }
  
  
  const getChatMeta = (client_id, public_key, socket) => {
    try {
      const chatKey = `${public_key}_${client_id}`;
  
      // Step 1: Fetch chat session and the participant who joined first
      const sessionAndUsersQuery = `
        SELECT cs.session_id, cs.served_by, cp.user_id, cp.join_time
        FROM chat_sessions cs
        JOIN chat_participants cp ON cs.session_id = cp.session_id
        WHERE cs.chat_key = ? 
          AND cs.is_active = 1 
          AND cp.role = 'admin'
        ORDER BY cp.join_time ASC
        LIMIT 1
      `;
  
      connection.query(sessionAndUsersQuery, [chatKey], (err, sessionAndUsers) => {
        if (err) {
          console.error("Error fetching chat session:", err);
          socket.emit("error", { message: "Error fetching chat session" });
          return;
        }
  
        // If no active chat session is found
        if (sessionAndUsers.length === 0) {
          socket.emit('served by', { isServed: false, message: 'No active chat session found' });
          return;
        }
  
        const { session_id, served_by, user_id } = sessionAndUsers[0];
  
        // Step 2: Fetch admin's name and designation using `user_id`
        const adminQuery = `
          SELECT u.real_name, u.designation 
          FROM users u 
          WHERE u.id = ? LIMIT 1
        `;
  
        connection.query(adminQuery, [user_id], (err, admin) => {
          if (err) {
            console.error("Error fetching admin details:", err);
            socket.emit("error", { message: "Error fetching admin details" });
            return;
          }
  
          if (admin.length === 0) {
            socket.emit('served by', { isServed: false, message: 'Served admin not found' });
            return;
          }
  
          const { real_name, designation } = admin[0];
  
          // Step 3: Fetch logo from the brands table using `public_key`
          const brandQuery = `
            SELECT logo 
            FROM brands 
            WHERE public_key = ? LIMIT 1
          `;
  
          connection.query(brandQuery, [public_key], (err, brand) => {
            if (err) {
              console.error("Error fetching brand logo:", err);
              socket.emit("error", { message: "Error fetching brand logo" });
              return;
            }
  
            const logoPath = brand.length > 0 ? brand[0].logo : null;
            const logourl = logoPath
              ? `https://backend.vchat.ourprojectdemo.com:3001/${logoPath}`
              : null;
  
            // Step 4: Check if rating exists for the session
            const ratingQuery = `
              SELECT rating 
              FROM chat_rate 
              WHERE chat_id = ? LIMIT 1
            `;
  
            connection.query(ratingQuery, [session_id], (err, ratingResult) => {
              if (err) {
                console.error("Error fetching rating:", err);
                socket.emit("error", { message: "Error fetching rating" });
                return;
              }
  
              // Check if a rating has already been submitted for this session
              const ratingExists = ratingResult.length > 0;
  
              // Compile visitor status and rating data
              const visitorStatus = {
                isServed: true,
                served_by: real_name,
                designation: designation,
                logo: logourl,
                rating: ratingExists ? ratingResult[0].rating : 0, // if rating exists, send the rating value
              };
  
              // Emit all necessary data
              socket.emit("served by", visitorStatus);
            });
          });
        });
      });
    } catch (error) {
      console.error("Error fetching chat history:", error);
      socket.emit("error", { message: "Error fetching chat history" });
    }
  };
  
  // Function to check if rating is given for a chat session
  const checkIfRatingGiven = (client_id, chatKey, socket) => {
    // Step 1: Fetch session_id from chat_sessions using chatKey
    const getSessionQuery = `SELECT session_id FROM chat_sessions WHERE chat_key = ? AND is_active = 1 LIMIT 1`;
  
    connection.query(getSessionQuery, [chatKey], (err, sessionResults) => {
        if (err) {
            console.error("Error fetching session:", err);
            return;
        }
  
        if (sessionResults.length > 0) {
            const sessionId = sessionResults[0].session_id;
  
            // Step 2: Check if rating exists in the chat_rate table
            const checkRatingQuery = `SELECT * FROM chat_rate WHERE chat_id = ? LIMIT 1`;
  
            connection.query(checkRatingQuery, [sessionId], (err, ratingResults) => {
                if (err) {
                    console.error("Error checking rating:", err);
                    return;
                }
  
                // Step 3: Emit result to the client based on whether a rating exists
                if (ratingResults.length > 0) {
                    socket.emit("rating chat", { rating: true });  // Rating is given
                 
                } else {
                    socket.emit("rating chat", { rating: false });  // Rating not given
                }
            });
        } else {
            socket.emit("rating", { rating: false });  // No active session found
        }
    });
  };
  function fetchClientDetails(client_id, socket) {
      // Assume you have a database connection available
      const query = 'SELECT name, email, phone FROM client WHERE client_id = ?';
  
      connection.query(query, [client_id], (error, results) => {
          if (error) {
              console.error('Error fetching client details:', error);
              return;
          }
  
          if (results.length > 0) {
              const clientDetails = results[0];
              // Emit the client details to the client socket
              socket.emit('client profile details', {
                  name: clientDetails.name,
                  email: clientDetails.email,
                  phone: clientDetails.phone
              });
          }  
      });
  }
  function fetchUnseenMessagesForAdmins(connection, public_key, client_id, io, chatKey) {
    // Step 1: Check for active chat session
    const activeChatSessionQuery = `
      SELECT session_id 
      FROM chat_sessions 
      WHERE chat_key = ? AND is_active = 1
    `;
  
    connection.query(activeChatSessionQuery, [chatKey], (err, sessionResults) => {
      if (err) {
        console.error("Error checking active chat session:", err);
        return;
      }
      console.log(sessionResults)
      if (sessionResults.length === 0) {
        console.log("No active chat session found.");
        return;
      }
  
      const sessionId = sessionResults[0].session_id;
  
      // Step 2: Fetch admin user IDs from chat_participants
      const adminParticipantsQuery = `
        SELECT user_id 
        FROM chat_participants 
        WHERE session_id = ? AND role = 'admin'
      `;
  
      connection.query(adminParticipantsQuery, [sessionId], (err, adminResults) => {
        if (err) {
          console.error("Error fetching admin participants:", err);
          return;
        }
  
        // Extract user IDs
        const adminUserIds = adminResults.map(admin => admin.user_id);
  
        if (adminUserIds.length === 0) {
          console.log("No admins found for this session.");
          return;
        }
  
        // Step 3: Count unseen messages for all admins
        const unseenMessagesPromises = adminUserIds.map(userId => {
          return new Promise((resolve, reject) => {
            const unseenMessagesQuery = `
              SELECT COUNT(*) AS unseenCount 
              FROM chat_messages 
              WHERE session_id = ? AND sender_id = ? AND is_seen = 0 AND type !='comment'
            `;
  
            connection.query(unseenMessagesQuery, [sessionId, userId], (err, countResults) => {
              if (err) {
                reject(err);
              } else {
                resolve(countResults[0].unseenCount);
              }
            });
          });
        });
  
        // Resolve all unseen message counts
        Promise.all(unseenMessagesPromises)
          .then(unseenCounts => {
            // Combine unseen counts
            const totalUnseenMessages = unseenCounts.reduce((total, count) => total + count, 0);
  
            // Emit the total unseen messages count to the client
            io.to(chatKey).emit("unseen messages from admin", {
              unseenMessages: totalUnseenMessages,
              client_id: client_id,
            });
  
            console.log(`Emitting total unseen messages to client ${client_id}: ${totalUnseenMessages}`);
          })
          .catch(err => {
            console.error("Error counting unseen messages:", err);
          });
      });
    });
  }
  
module.exports = {
    handleDisconnectTimeout,
    handleStatusChange,
    notifyAssignedAdmins,
    fetchUnseenMessagesForAdmins,
    fetchClientDetails,
    checkIfRatingGiven,
    getChatMeta,
    checkAndEmitRating,
    getGeolocationData,
    getMessagesBySessionId,
    notifyChatParticipants,
    getParticipantsWithNames
};
