import { useState, useRef } from 'react';
import Modal from 'react-modal';

export default function MessageFile({ message }) {
  const [isHovered, setIsHovered] = useState(false);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [zoomLevel, setZoomLevel] = useState(1);
  const [isDragging, setIsDragging] = useState(false);
  const [position, setPosition] = useState({ x: 0, y: 0 });
  const imageRef = useRef(null);
  const [lastMousePosition, setLastMousePosition] = useState({ x: 0, y: 0 });

  const handleMouseEnter = () => setIsHovered(true);
  const handleMouseLeave = () => setIsHovered(false);
  const handleImageClick = () => setIsModalOpen(true);
  const handleCloseModal = () => {
    setIsModalOpen(false);
    setZoomLevel(1);
    setPosition({ x: 0, y: 0 });
  };

  // Mouse wheel zoom functionality
  const handleWheel = (e) => {
    e.preventDefault();
    const newZoomLevel = zoomLevel + e.deltaY * -0.01;
    setZoomLevel(Math.min(Math.max(newZoomLevel, 1), 4)); // Zoom between 1x and 4x
  };

  // Drag to move functionality
  const handleMouseDown = (e) => {
    if (zoomLevel > 1) {
      setIsDragging(true);
      setLastMousePosition({ x: e.clientX, y: e.clientY });
    }
  };

  const handleMouseMove = (e) => {
    if (isDragging) {
      const dx = e.clientX - lastMousePosition.x;
      const dy = e.clientY - lastMousePosition.y;
      setPosition((prev) => ({
        x: prev.x + dx,
        y: prev.y + dy,
      }));
      setLastMousePosition({ x: e.clientX, y: e.clientY });
    }
  };

  const handleMouseUp = () => {
    setIsDragging(false);
  };

  const handleMouseLeaveModal = () => {
    setIsDragging(false);
  };

  // Double click to zoom in/out functionality
  const handleDoubleClick = () => {
    setZoomLevel(zoomLevel === 1 ? 2 : 1);
    if (zoomLevel === 1) {
      setPosition({ x: 0, y: 0 });
    }
  };

  const handleDownload = async () => {
    const imageUrl = `${process.env.NEXT_PUBLIC_API_URL}/${message.msg}`;
    try {
      const response = await fetch(imageUrl);
      const blob = await response.blob();
      const link = document.createElement('a');
      link.href = window.URL.createObjectURL(blob);
      link.download = message.msg.split('/').pop(); // Extract file name for download
      link.click();
    } catch (error) {
      console.error('Download failed', error);
    }
  };

  return (
    <div className="message-file" onMouseEnter={handleMouseEnter} onMouseLeave={handleMouseLeave}>
      <img
        src={`${process.env.NEXT_PUBLIC_API_URL3}/${message.msg}`}
        // src={`${process.env.NEXT_PUBLIC_API_URL}/${message.msg}`}
        alt="uploaded file"
        style={{ maxWidth: '200px', borderRadius: '5px', cursor: 'pointer' }}
        onClick={handleImageClick}
      />

      {/* Hover icon */}
      {isHovered && (
        <div className="hover-icon" onClick={handleImageClick}>
          🔍
        </div>
      )}

      {/* Modal to show image in full-screen */}
      <Modal
        isOpen={isModalOpen}
        onRequestClose={handleCloseModal}
        className="image-modal"
        overlayClassName="image-modal-overlay"
        onMouseLeave={handleMouseLeaveModal}
      >
        <div className="image-container">
          <img
            ref={imageRef}
            src={`${process.env.NEXT_PUBLIC_API_URL}/${message.msg}`}
            alt="uploaded file full screen"
            className="full-screen-image"
            style={{
              transform: `scale(${zoomLevel}) translate(${position.x}px, ${position.y}px)`,
              cursor: zoomLevel > 1 ? 'grab' : 'zoom-in',
            }}
            onDoubleClick={handleDoubleClick}
            onWheel={handleWheel}
            onMouseDown={handleMouseDown}
            onMouseMove={handleMouseMove}
            onMouseUp={handleMouseUp}
          />
        </div>
        <div className="modal-buttons">
          <button onClick={handleDownload} className="download-button">Download</button>
          <button onClick={handleCloseModal} className="close-button">Close</button>
        </div>
      </Modal>

      <style jsx>{`
        .message-file {
          position: relative;
          display: inline-block;
        }
        .hover-icon {
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%);
          font-size: 24px;
          color: white;
          background-color: rgba(0, 0, 0, 0.5);
          border-radius: 50%;
          padding: 10px;
          cursor: pointer;
        }
        .image-modal-overlay {
          background-color: rgba(0, 0, 0, 0.75);
        }
        .image-container {
          display: flex;
          justify-content: center;
          align-items: center;
          height: 100vh;
        }
        .full-screen-image {
          max-width: 100%;
          max-height: 100vh;
          object-fit: contain;
          transition: transform 0.2s ease;
        }
        .modal-buttons {
          display: flex;
          justify-content: space-between;
          position: absolute;
          bottom: 20px;
          left: 20px;
          right: 20px;
        }
        .download-button, .close-button {
          background: rgba(0, 0, 0, 0.5);
          color: white;
          border: none;
          border-radius: 5px;
          padding: 10px 20px;
          cursor: pointer;
        }
        .download-button:hover, .close-button:hover {
          background: rgba(0, 0, 0, 0.7);
        }
      `}</style>
    </div>
  );
}
