import apiRequest from '@/app/lib/InterceptorAxios/axios';
import React, { useState, useEffect } from 'react';
import swal from 'sweetalert';

const AutoReplyModel = ({ brandId, setAutoReplyOpenModel }) => {
  const [autoReplyData, setAutoReplyData] = useState({
    autoReply1: '',
    autoReply2: '',
    isActive1: true,
    isActive2: true,
    isFormSend: true,
    ignore_users_type: { admin: 0, marketing: 0, sales: 0 } // Default values
  });

  const [loading, setLoading] = useState(true);
  const [isExisting, setIsExisting] = useState(false);

  useEffect(() => {
    const fetchAutoReplyData = async () => {
      setLoading(true);
      try {
        const data = await apiRequest(`/api/autoReply/${brandId}`, 'get');
        
        if (data) {
          setAutoReplyData({
            autoReply1: data.auto_reply1 || '',
            autoReply2: data.auto_reply2 || '',
            isActive1: data.is_active1 ?? true,
            isActive2: data.is_active2 ?? true,
            isFormSend: data.is_form_send ?? true,
            ignore_users_type: {
              admin: data.ignore_users_type?.admin ?? 0,
              marketing: data.ignore_users_type?.marketing ?? 0,
              sales: data.ignore_users_type?.sales ?? 0
            }
          });
          setIsExisting(true);
        }
      } catch (error) {
        console.error('Error fetching auto-reply data:', error);
      } finally {
        setLoading(false);
      }
    };

    if (brandId) {
      fetchAutoReplyData();
    }
  }, [brandId]);

  const handleAutoReplyChange = (e) => {
    const { name, value } = e.target;
    setAutoReplyData((prevData) => ({
      ...prevData,
      [name]: value
    }));
  };

  const toggleAutoReplyStatus = (reply) => {
    setAutoReplyData((prevData) => ({
      ...prevData,
      [reply]: !prevData[reply]
    }));
  };

  const handleIgnoreRoleChange = (role) => {
    setAutoReplyData((prevData) => ({
      ...prevData,
      ignore_users_type: {
        ...prevData.ignore_users_type,
        [role]: prevData.ignore_users_type[role] === 1 ? 0 : 1
      }
    }));
  };

  const handleAutoReplySubmit = async (e) => {
    e.preventDefault();
    try {
      const method = isExisting ? 'put' : 'post';
      await apiRequest('/api/autoReply', method, {
        brand_id: brandId,
        auto_reply1: autoReplyData.autoReply1,
        is_active1: autoReplyData.isActive1,
        auto_reply2: autoReplyData.autoReply2,
        is_active2: autoReplyData.isActive2,
        is_form_send: autoReplyData.isFormSend,
        ignore_users_type: autoReplyData.ignore_users_type
      });
      swal('Success', `Auto-reply ${isExisting ? 'updated' : 'created'} successfully`, 'success');
      setAutoReplyOpenModel(false);
    } catch (error) {
      console.error('Error updating auto-reply settings:', error);
    }
  };

  return (
    <div className="modal-overlay-comp">
      <div className="modal-brand">
        <div className="modal-header-comp">
          <h5 className="modal-title">Auto Replies</h5>
          <button type="button" className="modal-close" onClick={() => setAutoReplyOpenModel(false)}>
            &times;
          </button>
        </div>

        <div className="modal-body">
          {loading ? (
            <div className="loading-bar">
              <div className="progress">
                <div className="progress-bar progress-bar-striped progress-bar-animated" style={{ width: '100%' }}>
                  Loading...
                </div>
              </div>
            </div>
          ) : (
            <form onSubmit={handleAutoReplySubmit}>
              <div className="form-group">
                <label htmlFor="autoReply1" className="autoReply-Model-label">
                  Auto Reply 1:
                  <div className="form-check form-switch auto-reply-switch">
                    <input
                      className="form-check-input auto-reply-model-input"
                      type="checkbox"
                      checked={autoReplyData.isActive1}
                      onChange={() => toggleAutoReplyStatus('isActive1')}
                    />
                  </div>
                </label>
                <textarea
                  className="form-control"
                  id="autoReply1"
                  name="autoReply1"
                  value={autoReplyData.autoReply1}
                  onChange={handleAutoReplyChange}
                  required
                />
              </div>

              <div className="form-group">
                <label className="autoReply-Model-label" htmlFor="autoReply2">
                  Auto Reply 2:
                  <div className="form-check form-switch auto-reply-switch">
                    <input
                      className="form-check-input auto-reply-model-input"
                      type="checkbox"
                      checked={autoReplyData.isActive2}
                      onChange={() => toggleAutoReplyStatus('isActive2')}
                    />
                  </div>
                </label>
                <textarea
                  className="form-control"
                  id="autoReply2"
                  name="autoReply2"
                  value={autoReplyData.autoReply2}
                  onChange={handleAutoReplyChange}
                  required
                />
              </div>

              <div className="form-group">
                <label className="autoReply-Model-label">Ignore Auto Replies For User Types:</label>
                <div className="d-flex auto-reply-ignore-container">
                  {['admin', 'marketing', 'sales'].map((role) => (
                    <div key={role} className="form-check auto-reply-ignore-box">
                      <input
                        className="form-check-input auto-reply-ignore-input"
                        type="checkbox"
                        id={`ignore${role}`}
                        checked={autoReplyData.ignore_users_type[role] === 1}
                        onChange={() => handleIgnoreRoleChange(role)}
                      />
                      <label className="form-check-label auto-reply-ignore-label" htmlFor={`ignore${role}`}>
                        {role.charAt(0).toUpperCase() + role.slice(1)}
                      </label>
                    </div>
                  ))}
                </div>
                <div className="form-group">
                <label className="autoReply-Model-label">Send Form:</label>
                <div className="form-check form-switch auto-reply-switch">
                  <input
                    className="form-check-input auto-reply-model-input"
                    type="checkbox"
                    checked={autoReplyData.isFormSend}
                    onChange={() => toggleAutoReplyStatus('isFormSend')}
                  />
                </div>
              </div>
              </div>

              <button type="submit" className="btn submit-btn p-btn mt-3 w-100">
                Save
              </button>
            </form>
          )}
        </div>
      </div>
    </div>
  );
};

export default AutoReplyModel;
