import React, { useRef, useEffect } from "react";

const ResizableTypingArea = ({ typingUsers }) => {
  const typingAreaRef = useRef(null);
  const textAreaRef = useRef(null)
  const startYRef = useRef(0);
  const startHeightRef = useRef(0);

  useEffect(() => {
    // Scroll the text area to the bottom whenever typingUsers updates
    if (textAreaRef.current) {
        textAreaRef.current.scrollTop = textAreaRef.current.scrollHeight;
    }
  }, [typingUsers]);

  const startResizing = (e) => {
    if (e.button !== 0) return; // Only allow left-click

    // Capture initial mouse position and element height
    startYRef.current = e.clientY;
    const rect = typingAreaRef.current.getBoundingClientRect();
    startHeightRef.current = rect.height;

    // Attach mousemove and mouseup listeners
    document.addEventListener("mousemove", handleResizing);
    document.addEventListener("mouseup", stopResizing);

    // Disable text selection during resizing
    document.body.style.userSelect = "none";
  };

  const handleResizing = (e) => {
    // Calculate the new height dynamically
    const deltaY = e.clientY - startYRef.current;
    const newHeight = Math.max(startHeightRef.current - deltaY, 50); // Prevent height from going below 50px

    // Apply the new height to the typing area
    if (typingAreaRef.current) {
      typingAreaRef.current.style.height = `${newHeight}px`;
    }
  };

  const stopResizing = () => {
    // Remove event listeners
    document.removeEventListener("mousemove", handleResizing);
    document.removeEventListener("mouseup", stopResizing);

    // Re-enable text selection
    document.body.style.userSelect = "";
  };

  return (
    <div
      className="visitor-typing-area"
      title="Visitor typing area"
      ref={typingAreaRef}
      style={{
        height: "100px",
        border: "1px solid #ccc",
        position: "relative",
        
        resize: "none",
      }}
    >
      {/* Resize Handle */}
      <div
        className="resize-handle"
        onMouseDown={startResizing}
        style={{
          position: "absolute",
          bottom: 0,
          left: 0,
          right: 0,
          height: "7px",
          cursor: "row-resize",
          backgroundColor: "#e0e0e0",
        }}
        title="Resize"
      ></div>

      {/* Typing Indicator */}
      {typingUsers && typingUsers.size > 0 && (
        <div className="typing-indicator">
          {Array.from(typingUsers.entries()).map(([userId, text], index) => (
            <div key={index} className="typing-user">
             
              <textarea
                className="typing-textarea"
                ref={textAreaRef}
                readOnly
                style={{ width: "100%", overflow: "auto" }}
                value={
                  text || "typing..."
                     
                }
              ></textarea>
            </div>
          ))}
        </div>
      )}
    </div>
  );
};

export default ResizableTypingArea;
