import CryptoJS from 'crypto-js'; // Install with `npm install crypto-js`

const getDeviceId = async () => {
  const detectIncognitoMode = async () => {
    const fs = window.RequestFileSystem || window.webkitRequestFileSystem;
    if (!fs) {
      return false; // Incognito detection not supported
    }
    return new Promise((resolve) => {
      fs(window.TEMPORARY, 100, () => resolve(false), () => resolve(true));
    });
  };

  const generateFingerprint = async () => {
    const userAgent = navigator.userAgent;
    const language = navigator.language;
    const vendor = navigator.vendor;
    const platform = navigator.platform;
    const colorDepth = screen.colorDepth;
    const pixelDepth = screen.pixelDepth;
    const timezoneOffset = new Date().getTimezoneOffset();
    const plugins = Array.from(navigator.plugins)
      .map((plugin) => plugin.name)
      .join(',');

    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    ctx.textBaseline = 'top';
    ctx.font = '16px Arial';
    ctx.fillStyle = 'blue';
    ctx.fillText(userAgent, 10, 50);
    const canvasFingerprint = canvas.toDataURL();

    return [
      userAgent,
      language,
      vendor,
      platform,
      colorDepth,
      pixelDepth,
      timezoneOffset,
      plugins,
      canvasFingerprint,
    ].join('::');
  };

  const hashWithFallback = async (input) => {
    try {
      if (window.crypto && crypto.subtle) {
        const encoder = new TextEncoder();
        const data = encoder.encode(input);
        const hashBuffer = await crypto.subtle.digest('SHA-256', data);
        return Array.from(new Uint8Array(hashBuffer))
          .map((byte) => byte.toString(16).padStart(2, '0'))
          .join('');
      } else {
        // Fallback to CryptoJS if Web Crypto API is not available
        return CryptoJS.SHA256(input).toString(CryptoJS.enc.Hex);
      }
    } catch (error) {
      console.error('Hashing failed:', error);
      throw new Error('Unable to hash input');
    }
  };

  const storedId = localStorage.getItem('visitorId');
  if (storedId) {
    return storedId; // Return stored ID if available
  }

  const fingerprint = await generateFingerprint();
  const isIncognito = await detectIncognitoMode();

  const randomSalt = Math.random().toString(36).substring(2, 15);
  const uniqueId = await hashWithFallback(fingerprint + randomSalt);

  // Append "-incognito" if incognito mode is detected
  const finalId = isIncognito ? `${uniqueId}-incognito` : uniqueId;

  localStorage.setItem('visitorId', finalId);
  return finalId;
};

export default getDeviceId;
 
